import java.util.HashSet;
/**
 *Graph.java is a class to assist with graph and network algorithms for CITS2200.
 *Graphs may be directed or not, weighted or not.
 *Graphs are simple: no loop edges (v,v) and no multiple edges between nodes.
 *Vertices are labeled from 0 to number of vertices - 1.
 *Edge weights must be positive integers (of value 1 if the graph is not weighted).
 *Utility methods are provided to create random graphs.
 *This implementation uses an adjacency matrix to represent the graph.
 *@author Tim French modified Rachel Cardell-Oliver
 **/

public class AdjacencyMatrixGraph implements GraphADT{

	private int[][] edgeMatrix;
	private int numberOfVertices;
	private boolean directed;
	private boolean weighted;

	/**
	 *Constructs an graph with n vertices and no edges
	 *vertices are labelled 0 to n-1
	 *@param vertices the number of vertices (must be greater than 0)
	 *@param weighted true if the graph is to be weighted
	 *@param directed true if the graph is directed
	 **/
	public AdjacencyMatrixGraph(int vertices, boolean weighted, boolean directed){
		if(vertices<1) numberOfVertices = 0;
		else numberOfVertices = vertices;
		edgeMatrix = new int[numberOfVertices][numberOfVertices];
		this.weighted = weighted;
		this.directed = directed;
	}

	/**
	 *Constructs an unweighted, undirected graph with n vertices and no edges
	 *vertices are labelled 0 to vertices-1
	 *@param vertices the number of vertices (must be greater than 0)
	 **/
	public AdjacencyMatrixGraph(int vertices){
		this(vertices, false, false);
	}


	/**
	 *Adds an edge of the given weight to the graph.
	 *If the graph is undirected the reverse edge is also added.
	 *If the graph is unweighted the weight is set to one.
	 *If an edge already exists it is overwritten with the new weight.
	 *Only simple graphs allowed (no loop edges v,v)
	 *@param u the start of the edge
	 *@param v the end of the edge
	 *@param weight the weight of the edge
	 **/
	public void addEdge(int u, int v, int weight){
		if (0 <= u && 0 <= v && u < numberOfVertices && v < numberOfVertices && u!=v) {
			if(weight<1) weight = 0;
			else if(!weighted) weight = 1;
			edgeMatrix[u][v] = weight;
			if(!directed) edgeMatrix[v][u] = weight;
		}
		else throw new RuntimeException("Vertex out of bounds");
	}

	/**
	 *Adds an edge of weight 1 to the graph.
	 *If the graph is undirected the reverse edge is also added.
	 *@param u the start of the edge
	 *@param v the end of the edge
	 **/
	public void addEdge(int u, int v){
		this.addEdge(u,v,1);
	}

	/**
	 *Removes the edge from the graph (sets the edge weight to 0).
	 *If the graph is undirected the reverse edge is also removed.
	 *@param u the start of the edge
	 *@param v the end of the edge
	 **/
	public void removeEdge(int u, int v){
		this.addEdge(u,v,0);
	}

	/**
	 * vertex IDs are 0 to numberOfVertices-1
	 *@return the weight of an edge.
	 *@param u the start of the edge
	 *@param v the end of the edge
	 **/
	public int getEdgeWeight(int u, int v){
		if (0 <= u && 0 <= v && u < numberOfVertices && v < numberOfVertices)
			return edgeMatrix[u][v];
		else throw new RuntimeException("Vertex out of bounds");
	}
	
	/**
	 *@return true if u and v are adjacent
	 *@param u the start of the edge
	 *@param v the end of the edge
	 **/
	public boolean isAdjacent(int u, int v){
		return (edgeMatrix[u][v] != 0);
	}
	
	/**
	 *@return an array list of the vertex ids of the neighbours of v
	 *@param v the vertice
	 **/
	public HashSet<Integer> getNeighbours(int v) {
		int[] neighbours = (int[])edgeMatrix[v].clone();
		HashSet<Integer> nlist = new HashSet<Integer>();

		for (int i=0; i < numberOfVertices; i++) {
			if (neighbours[i]>0) { nlist.add(i); }
		}
		return (nlist);
	}

	/**
	 *@return the number of vertices
	 **/
	public int getNumberOfVertices(){
		return numberOfVertices;
	}

	/**
	 *@return true if the graph is directed
	 **/
	public boolean isDirected(){
		return directed;
	}

	/**
	 *@return true if the graph is weighted
	 **/
	public boolean isWeighted(){
		return weighted;
	}
	
	/**
	 *Count distinct edges in a directed or undirected graph
	 *Note this operation runs in time n^2.
	 *@return int number of non-zero entries in the edge matrix
	 **/
	public int getNumberOfEdges(){
		int count = 0;
		if (directed) {
			for(int i = 0; i<edgeMatrix.length; i++)
				for (int j=0; j<edgeMatrix[i].length; j++)
					if (edgeMatrix[i][j]>0) { count++; }
		} else { //undirected graph
			for(int i = 0; i<edgeMatrix.length; i++)
				for (int j=i+1; j<edgeMatrix[i].length; j++)
					if (edgeMatrix[i][j]>0) { count++; }
		}
		return count;
	}
	
	/**
	 *This method should not be used to write graph algorithms
	 * use the graphADT operations instead
	 *Note this operation runs in time n^2.
	 *To find a single edge use getEdgeWeight.
	 *@return a clone of the edgeMatrix
	 **/
	private int[][] getEdgeMatrix(){
		int[][] clone = new int[edgeMatrix.length][edgeMatrix.length];
		for(int i = 0; i<edgeMatrix.length; i++)
			clone[i] = (int[])edgeMatrix[i].clone();
		return clone;	
	}

	/**
	 *Utility method to create a random graph 
	 *the graph is simple (no loop edges v,v)
	 *@return a random graph of the specified density
	 *@param numberOfVertices the number of vertices
	 *@param maxWeight the maximum edgeWeight for the graph.
	 *@param density between 0 and 1, the probability of an edge existing.
	 **/
	public static AdjacencyMatrixGraph randomGraph(int numberOfVertices, boolean weighted, boolean directed, double density, int maxWeight){
		AdjacencyMatrixGraph g = new AdjacencyMatrixGraph(numberOfVertices, weighted, directed);
		for(int i = 0; i<g.getNumberOfVertices(); i++){
			for(int j = 0; j<(directed?g.getNumberOfVertices():i); j++){
				if(Math.random()<density && i!=j){
					if (!weighted) {
						g.edgeMatrix[i][j] = 1;
					} else {
						g.edgeMatrix[i][j] = (int)(Math.random()*(maxWeight-1))+1;
					}
					if(!directed) g.edgeMatrix[j][i] = 1;
				}
			}
		}
		return g;
	}

	/**
	 *Creates a random, unweighted, undirected graph
	 *@return a random graph of the specified density
	 *@param numberOfVertices the nukmber of vertices
	 *@param directed true if the graph is to be directed
	 *@param density between 0 and 1, the probablity of an edge existing.
	 **/
	public static AdjacencyMatrixGraph randomGraph(int numberOfVertices, double density){
		return randomGraph(numberOfVertices, false, false, density, 1);
	}
	
	/**
	 *Creates a random weighted, directed graph. 
	 * The weights will be even distributed between 1 and
	 * the maximum edge weight (inclusive).
	 *@return a random graph of the specified density
	 *@param numberOfVertices the number of vertices
	 *@param directed true if the graph is to be directed
	 *@param density between 0 and 1, the probability of an edge existing.
	 *@param maxWeight the maximum edgeWeight for the Graph.
	 **/
	public static AdjacencyMatrixGraph randomGraph(int numberOfVertices, double density, int maxWeight){
		return randomGraph(numberOfVertices, true, true, density, maxWeight);
	}
	

	/**
	 *This method produces a representation of the
	 *graph that corresponds to the adjacency 
	 *matrix used by the readFile method.
	 *@return a String representation of the graph,
	 **/
	public String toString(){
		StringBuffer s = new StringBuffer(getNumberOfVertices()+"\n");
		for(int i = 0; i<numberOfVertices; i++){
			for(int j = 0; j<numberOfVertices; j++){
				s.append(edgeMatrix[i][j]);
				s.append("\t");
			}
			s.append("\n");
		}
		return s.toString();
	}

}


