import java.util.Arrays;
import java.util.NoSuchElementException;

import CITS2200.Overflow;

/**
 * A Priority Queue of Strings using the Binary Heap data structure
 * where priority is determined by the lexicographical order of the Strings (ie dictionary order)
 * See Wiess Chapter 21 for a more general version
 * @author rachelcardell-oliver
 *
 */
public class BinaryHeapPriorityQueue
{

	public static int DEFAULT_CAPACITY = 100;
	public boolean demo = true; //or false to supress explanatory messages
	
	private int currentSize; //number of elements currently in the priority queue
	private String[] arr;   //array of Strings for the heap
	
	public BinaryHeapPriorityQueue() {
		currentSize = 0;
		arr = new String[DEFAULT_CAPACITY + 1];
	}
	
	/**
     * Make this PriorityQueue empty.
     */
    public void clear() {
        currentSize = 0;
    }
    
    public int size() {
        return(currentSize);
    }
    
	public boolean isEmpty() {
		return(currentSize == 0);
	}
    
	public boolean isFull () {
		return(currentSize + 1 >= arr.length);
	}
	
	/**
     * Establish heap order property from an arbitrary
     * arrangement of items in arr. Runs in linear time.
     */
    public void buildHeap()
    {
        for( int i = currentSize / 2; i > 0; i-- )
            percolateDown( i );
    }
	
    private int compare(String s1,String s2) {
    	return s1.compareTo(s2);
    	//alternatively, use String.CASE_INSENSITIVE_ORDER.compare(s1,s2)
    }
    
    /**
     * Internal method to percolate down in the heap.
     * @param hole the index at which the percolate begins.
     Also known as bubbledown
     */
    private void percolateDown( int hole )
    {
        int child;
        String tmp = arr[ hole ];
        if (demo) { System.out.println("Percolate from "+hole); }

        for( ; hole * 2 <= currentSize; hole = child )
        {
            child = hole * 2;
            if( child != currentSize &&
            		compare( arr[ child + 1 ], arr[ child ] ) < 0 )
                child++;
            if( compare( arr[ child ], tmp ) < 0 ) {
                arr[ hole ] = arr[ child ];
            	if (demo) { System.out.println("Move "+arr[child]+" at "+child+ " to "+hole); }
            } else {
                break;
            }
        }
        arr[ hole ] = tmp;
        if (demo) { System.out.println("Place "+tmp+" into "+hole); }
    }
    
	/**
	 * adds an item to the priority queue 
	 * with priority based on the lexicographical order of strings
	 * @param a String item to be added
	 */

	public void enqueue(String a) {
		if(isFull()) throw new Overflow("Full!");
		//alternatively see Weiss for doubleArray method to create more space

        //create a hole at the end and wait the element to insert at 0
        int hole = ++currentSize;
        arr[ 0 ] = a;
        if (demo) { System.out.println("Enqueue "+a+" into heap with "+ currentSize + " elements"); }
        
        // Percolate up (bubble up) - to find where a should be inserted
        for( ; compare( a, arr[ hole / 2 ] ) < 0; hole /= 2 ) {
            arr[ hole ] = arr[ hole / 2 ];
            int h2=hole/2;
            if (demo) { System.out.println("move "+arr[ hole / 2 ]+" at "+ h2 + " to "+hole); }
        }
        //then insert a in the correct position
        arr[ hole ] = a;
        if (demo) { System.out.println("insert "+a+" at "+ hole); }
	}

	/**
     * Removes the smallest item in the priority queue.
     * @return the smallest item.
     * @throws NoSuchElementException if empty.
     */
	public String dequeue() {
        if (isEmpty()) throw new NoSuchElementException("Nothing to dequeue");
        //otherwise return the head element and restructure the heap with percolate down
        String minItem = arr[ 1 ];
        arr[ 1 ] = arr[ currentSize-- ];
        percolateDown(1);
        return( minItem );
    }
	

	/** 
	 * examine the value of the highest priority element (at front)
	 * this operation is O(1)
	 * @return element value of the front element
	 * @throws Exception must be called from try catch
	 */
	public String examine()  {
		if (isEmpty()) {
			throw new NoSuchElementException("Empty Queue");
		} else {
			return (arr[1]);
		}
	}
	
	
	// TODO where are the pir elements
	/** display the contents of the priority queue array
	 * (as stored, rather than in pri order)
	 */
	public String toString() {
		return(Arrays.toString(arr));
		/*
		String a1 = "";
		for (int i=0; i < arr.length; i++) {
			a1 = a1 + arr[i];
		}
		return(a1);
		*/
	}
	
	
}
