#include <cnet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//  This is an implementation of a stop-and-wait data link protocol,
//  providing a reliable data-link layer for a 2-node network.
//  This protocol employs only data and acknowledgement frames -
//  piggybacking and negative acknowledgements are not used.
//
//  It is based on Tanenbaum's 'protocol 4', 2nd edition, p227.


//  A FRAME CAN BE EITHER DATA OR AN ACKNOWLEDGMENT FRAME
typedef enum {
    DL_DATA, DL_ACK
} FRAMEKIND;

//  THE FORMAT OF A FRAME HEADER
typedef struct {
    FRAMEKIND    kind;      	// only ever DL_DATA or DL_ACK
    size_t	 len;       	// the length of the PAYLOAD, only
    int          seq;       	// only ever 0 or 1
    int          checksum;  	// checksum of the HEADER and PAYLOAD
} HEADER;

//  DATA FRAMES CARRY A MAXIMUM-SIZED PAYLOAD, OUR DATA
typedef struct {
    char        data[MAX_MESSAGE_SIZE];
} PAYLOAD;

//  THE FORMAT OF A FRAME
typedef struct {
    HEADER	header;
    PAYLOAD	payload;
} FRAME;

//  SOME HELPFUL MACROS FOR COMMON CALCULATIONS
#define FRAME_SIZE(frame)	(sizeof(HEADER) + frame.header.len)
#define increment(seq)		seq = 1-seq

#define RHS                     "\t\t\t\t\t"

//  -------------------------------------------------------------------

//  STATE VARIABLES HOLDING SEQUENCE NUMBERS
int		nextdatatosend		= 0;    // by sender
int       	ackexpected		= 0;    // by sender
int		dataexpected		= 0;    // by receiver

//  STATE VARIABLES HOLDING INFORMATION ABOUT THE LAST MESSAGE
PAYLOAD       	lastdata;
size_t		lastdatalength		= 0;
CnetTimerID	lasttimer		= NULLTIMER;

CnetTime time_taken(int link, size_t nbytes)
{
    return nbytes*8 *
	(1000000 / OS->links[link].bandwidth) + OS->links[link].propagationdelay;
}

//  A FUNCTION TO TRANSMIT A FRAME - EITHER DATA OR AN ACKNOWLEDGMENT
void transmit_frame(FRAMEKIND kind, PAYLOAD *data, size_t length, int seqno)
{
    FRAME       frame;
    int		link = 1;

//  INITIALISE THE FRAME'S HEADER FIELDS
    frame.header.kind      = kind;
    frame.header.len       = length;	// length of ACK payload is 0
    frame.header.seq       = seqno;
    frame.header.checksum  = 0;

    switch (kind) {
    case DL_ACK :
        printf("ACK transmitted, seq=%i\n", seqno);
	break;

    case DL_DATA: {
        printf("DATA transmitted, seq=%i\n", seqno);
        memcpy(&frame.payload.data, data, length);

	CnetTime timeout =	time_taken(link, FRAME_SIZE(frame)) +
				time_taken(link, sizeof(HEADER)) ;

        lasttimer = CNET_start_timer(EV_TIMER1, 2 * timeout, 0);
        CNET_set_LED_on(0, 0, "red");
	break;
      }
    }

//  FINALLY, WRITE THE FRAME TO THE PHYSICAL LAYER
length -= 20;
    length			= FRAME_SIZE(frame);
    frame.header.checksum	= CNET_ccitt((unsigned char *)&frame, length);
    CHECK(CNET_write_physical(link, &frame, &length));
}

//  THE APPLICATION LAYER HAS A NEW MESSAGE TO BE DELIVERED
EVENT_HANDLER(application_ready)
{
    CnetAddr destaddr;

    lastdatalength  = sizeof(PAYLOAD);
    CHECK(CNET_read_application(&destaddr, &lastdata, &lastdatalength));
    CNET_disable_application(ALLNODES);

    printf("down from application, seq=%i\n", nextdatatosend);
    transmit_frame(DL_DATA, &lastdata, lastdatalength, nextdatatosend);
    increment(nextdatatosend);
}

//  PROCESS THE ARRIVAL OF A NEW FRAME, VERIFY CHECKSUM, ACT ON ITS FRAMEKIND
EVENT_HANDLER(physical_ready)
{
    FRAME        frame;
    size_t	 len = sizeof(FRAME);
    int          link, arriving_checksum;

//  RECEIVE THE NEW FRAME
    CHECK(CNET_read_physical(&link, &frame, &len));

//  CALCULATE THE CHECKSUM OF THE ARRIVING FRAME, IGNORE IF INVALID
    arriving_checksum		= frame.header.checksum;
    frame.header.checksum  	= 0;
    if(CNET_ccitt((unsigned char *)&frame, len) != arriving_checksum) {
        printf("%sBAD checksum - frame ignored\n", RHS);
        return;           // bad checksum, just ignore frame
    }

    switch (frame.header.kind) {
//  AN ACKNOWLEDGMENT HAS ARRIVED, ENSURE IT'S THE ACKNOWLEDGMENT WE WANT
    case DL_ACK :
        if(frame.header.seq == ackexpected) {
            printf("%sACK received, seq=%i\n", RHS, frame.header.seq);
            CNET_stop_timer(lasttimer);
            increment(ackexpected);
            CNET_enable_application(ALLNODES);
            CNET_set_LED_on(0, 0, "green");
        }
	break;

//  SOME DATA HAS ARRIVED, ENSURE IT'S THE ONE DATA WE EXPECT
    case DL_DATA :
        printf("%sDATA received, seq=%i, ", RHS, frame.header.seq);
        if(frame.header.seq == dataexpected) {
            printf("up to application\n");
            len = frame.header.len;
            CHECK(CNET_write_application(&frame.payload.data, &len));
            increment(dataexpected);
        }
        else
            printf("ignored\n");
        transmit_frame(DL_ACK, NULL, 0, frame.header.seq);  // ACK the data
	break;
    }
}

//  WHEN A TIMEOUT OCCURS, WE RE-TRANSMIT THE MOST RECENT DATA (MESSAGE)
EVENT_HANDLER(timeouts)
{
    printf("timeout, seq=%i\n", ackexpected);
    /* re- */ transmit_frame(DL_DATA, &lastdata, lastdatalength, ackexpected);
}

//  DISPLAY THE CURRENT SEQUENCE NUMBERS WHEN A BUTTON IS PRESSED
EVENT_HANDLER(showstate)
{
    printf(
    "\n\tackexpected\t= %i\n\tnextdatatosend\t= %i\n\tdataexpected\t= %i\n",
		    ackexpected, nextdatatosend, dataexpected);
}

//  -------------------------------------------------------------------

//  THIS FUNCTION IS CALLED ONCE, AT THE BEGINNING OF THE WHOLE SIMULATION
EVENT_HANDLER(reboot_node)
{
//  ENSURE THAT WE HAVE A 2-NODE NETWORK
    if(OS->nodenumber > 1) {
	fprintf(stderr, "This is not a 2-node network!\n");
	exit(EXIT_FAILURE);
    }

//  INDICATE THE EVENTS OF INTEREST FOR THIS PROTOCOL
    CHECK(CNET_set_handler( EV_APPLICATIONREADY, application_ready, NODATA));
    CHECK(CNET_set_handler( EV_PHYSICALREADY,    physical_ready, NODATA));
    CHECK(CNET_set_handler( EV_TIMER1,           timeouts, NODATA));

//  BIND A FUNCTION AND A LABEL TO ONE OF THE NODE'S BUTTONS
    CHECK(CNET_set_handler( EV_DEBUG0,           showstate, NODATA));
    CHECK(CNET_set_debug_string( EV_DEBUG0, "State" ));

//  THIS PROTOCOL IS CURRENTLY WRITTEN SO THAT ONLY node0 WILL GENERATE
//  AND TRANSMIT MESSAGES, AND node1 WILL RECEIVE THEM.
//  THIS RESTRICTION SIMPLIFIES UNDERSTANDING FOR THOSE NEW TO cnet.
//
//  THE RESTRICTION CAN EASILY BE REMOVED BY REMOVING THE 'if' TEST.
//  BOTH NODES WILL THEN TRANSMIT AND RECEIVE (WHY?)

//    if(OS->nodenumber == 0)
	CNET_enable_application(ALLNODES);

    CNET_set_LED_on(0, 0, "green");
}
