#include <cnet.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

//  This is an implementation of the GoBack-N data-link layer protocol.
//  Written by Chris.McDonald@uwa.edu.au

// ----------- ALL THIS SHOULD REALLY GO IN ANOTHER HEADER FILE --------------

#if	!defined(NBITS)
#define	NBITS			2       // default sender-window-size = 4
#endif

#define	NRBUFS			(1<<NBITS)

typedef	struct {		// PACKETs are managed by layer above DLL
    size_t	length;
    char	msg[MAX_MESSAGE_SIZE];
} PACKET;

#define PACKET_HEADER_SIZE      (sizeof(PACKET) - MAX_MESSAGE_SIZE)
#define PACKET_SIZE(p)          (PACKET_HEADER_SIZE + (p).length)


typedef enum    { DLL_DATA, DLL_ACK }   FRAMEKIND;

typedef struct {		// FRAMEs are managed by the Data Link Layer
    FRAMEKIND	kind;
    int		checksum;	// of the whole frame
    int		seqno;		// seqno of data or ack of this frame
    size_t	length;		// the length of the packet portion only
    PACKET	packet;
} FRAME;

#define	FRAME_HEADER_SIZE	(sizeof(FRAME) - sizeof(PACKET))
#define	FRAME_SIZE(f)		(FRAME_HEADER_SIZE + f.length)


// ---------------------------------------------------------------------------

PACKET		*outpackets;
CnetTimerID	*timers;

int		nextdatatosend	= 0;
int		dataexpected	= 0;
int		ackexpected	= 0;

int		nbuffered	= 0;

int inc(int seqno) { return (seqno + 1) % NRBUFS; }

//  A <= B < C  wrapped
bool between(int a, int b, int c)
{
    return (((a<=b) && (b< c)) || ((c< a) && (a<=b)) || ((b< c) && (c< a)));
}

// ---------------------------------------------------------------------------

//  TRANSMIT EITHER A DLL_DATA or DLL_ACK FRAME ON LINK 1
void transmit_frame(FRAMEKIND kind, PACKET *packet, size_t length, int seqno)
{
    FRAME       f;
    size_t	framelength;
    int		link = 1;

    f.kind	= kind;
    f.length	= length;
    f.seqno	= seqno;

    switch (f.kind) {
    case DLL_ACK :
	printf("DLL_ACK %i sent\n", seqno);
	break;

    case DLL_DATA : {
	printf("DLL_DATA %i sent\n", seqno);

	CnetTime arrives =
                FRAME_SIZE(f)*((CnetTime)8000000 / OS->links[link].bandwidth) +
                OS->links[link].propagationdelay;
	timers[seqno]	= CNET_start_timer(EV_TIMER1, 3 * arrives, (CnetData)seqno);

        memcpy(&f.packet, packet, length);
	break;
      }
    }

    framelength	= FRAME_SIZE(f);
    f.checksum	= 0;
    f.checksum	= CNET_ccitt((unsigned char *)&f, framelength);
    CHECK(CNET_write_physical(link, &f, &framelength));
}


//  READ A NEW MESSAGE FROM THE APPLICATION LAYER TO BE DELIVERED
EVENT_HANDLER(application_ready)
{
    CnetAddr	dest;
    int		nd2s	= nextdatatosend;

    outpackets[nd2s].length	= MAX_MESSAGE_SIZE;
    CHECK(CNET_read_application(&dest, outpackets[nd2s].msg, &outpackets[nd2s].length));
    printf("down from application %i\n", nd2s);

    if(++nbuffered == (NRBUFS-1)) {		// now out of buffer space!
	CNET_disable_application(ALLNODES);
    }
    transmit_frame(DLL_DATA, &outpackets[nd2s], PACKET_SIZE(outpackets[nd2s]), nd2s);
    nextdatatosend	= inc(nextdatatosend);
}


//  READ A DLL FRAME FROM THE PHYSICAL LAYER
EVENT_HANDLER(physical_ready)
{
    FRAME	frame;
    size_t	length;
    int		link, checksum;

    length		= sizeof(FRAME);
    CHECK(CNET_read_physical(&link, &frame, &length));

    checksum		= frame.checksum;
    frame.checksum	= 0;
    if(CNET_ccitt((unsigned char *)&frame, length) != checksum) {
	printf("\t\tBAD checksum\n");
	return;				        // bad checksum, ignore frame
    }

    switch (frame.kind) {
    case DLL_ACK :
        while( between(ackexpected, frame.seqno, nextdatatosend) ) {
	    CNET_stop_timer(timers[ackexpected]);
	    timers[ackexpected]	= NULLTIMER;
	    ackexpected		= inc(ackexpected);

	    --nbuffered;			// buffer now available
	    CNET_enable_application(ALLNODES);
        }
	break;

    case DLL_DATA : {
	if(frame.seqno == dataexpected) {
	    printf("\t\tDLL_DATA %i arrived\n", dataexpected);
            printf("\t\tto application %i\n", dataexpected);
            length          = frame.packet.length;
            CHECK(CNET_write_application( frame.packet.msg, &length));
            transmit_frame(DLL_ACK, NULL, 0, dataexpected);
            dataexpected    = inc(dataexpected);
	}
	else {
	    printf("\t\tDLL_DATA %i ignored\n", frame.seqno);
	}
	break;
      }
    }
}

//  RE-TRANSMIT ALL BUFFERED PACKETS STARTING FROM THE 1st ACK EXPECTED
EVENT_HANDLER(DLL_timeouts)
{
    nextdatatosend  = ackexpected;
    for(int n=0 ; n<nbuffered ; ++n) {
        int nd2s    = nextdatatosend;

        CNET_stop_timer(timers[nd2s]);
        transmit_frame(DLL_DATA, &outpackets[nd2s], PACKET_SIZE(outpackets[nd2s]), nd2s);
        nextdatatosend  = inc(nextdatatosend);
    }
}

//  ALLOCATE+INITIALIZE BUFFERS, DEFINE EVENT HANDLERS
EVENT_HANDLER(reboot_node)
{
    if(OS->nodenumber > 1) {
	fprintf(stderr,"This is not a 2-node network!\n");
	exit(EXIT_FAILURE);
    }

    outpackets		= calloc(NRBUFS, sizeof(PACKET));
    timers		= calloc(NRBUFS, sizeof(CnetTimerID));

    for(int b=0 ; b<NRBUFS ; b++) {
	timers[b]	= NULLTIMER;
    }

    CHECK(CNET_set_handler( EV_APPLICATIONREADY,application_ready,	0));
    CHECK(CNET_set_handler( EV_PHYSICALREADY,	physical_ready,		0));
    CHECK(CNET_set_handler( EV_TIMER1,		DLL_timeouts,		0));

// if(OS->nodenumber == 0)
    CNET_enable_application(ALLNODES);
}
