/*
 * Orthographic view of a color cube using LookAt() and Ortho()
 *
 * Colors are assigned to each vertex and then the rasterizer interpolates
 *   those colors across the triangles. Inspect the 'keyboard' function
 *   in the code. You can change the left, right, top, bottom, near, and far
 *   clipping planes by entering the appropriate letter on the keyboard.
 *
 * The current clipping plane values and the camera position are shown on the
 *   title bar of the window.
 *
 * Note that if the viewing distance of the camera is small, the front part of the
 *   might be clipped away as it falls in front of the clipping plane. You can
 *   also adjust the near and far clipping planes to see how clipping takes place.
 *
 * In this example, we have a cube and it would be best displayed with a clipping
 *   volume that has width = height. If the clipping volume has width <> height,
 *   then when it is mapped to a square viewport, it would distort the shape of
 *   the cube.
 *
 * The reshape callback function currently doesn't do anything.
 *
 * Revised by Du Huynh
 * School of Computer Science and Software Engineering
 * The University of Western Australia
 * April 2017
 */

#include <stdio.h>
#include "Angel.h"

#define WIN_WIDTH 700
#define WIN_HEIGHT 700

typedef Angel::vec4  color4;
typedef Angel::vec4  point4;

const int NumVertices = 36; //(6 faces)(2 triangles/face)(3 vertices/triangle)

point4 points[NumVertices];
color4 colors[NumVertices];

// Vertices of a unit cube centered at origin, sides aligned with axes
point4 vertices[8] = {
    point4( -0.5, -0.5,  0.5, 1.0 ),
    point4( -0.5,  0.5,  0.5, 1.0 ),
    point4(  0.5,  0.5,  0.5, 1.0 ),
    point4(  0.5, -0.5,  0.5, 1.0 ),
    point4( -0.5, -0.5, -0.5, 1.0 ),
    point4( -0.5,  0.5, -0.5, 1.0 ),
    point4(  0.5,  0.5, -0.5, 1.0 ),
    point4(  0.5, -0.5, -0.5, 1.0 )
};

// RGBA olors
color4 vertex_colors[8] = {
    color4( 0.0, 0.0, 0.0, 1.0 ),  // black
    color4( 1.0, 0.0, 0.0, 1.0 ),  // red
    color4( 1.0, 1.0, 0.0, 1.0 ),  // yellow
    color4( 0.0, 1.0, 0.0, 1.0 ),  // green
    color4( 0.0, 0.0, 1.0, 1.0 ),  // blue
    color4( 1.0, 0.0, 1.0, 1.0 ),  // magenta
    color4( 1.0, 1.0, 1.0, 1.0 ),  // white
    color4( 0.0, 1.0, 1.0, 1.0 )   // cyan
};

const GLfloat default_viewDist = 2.0;
const GLfloat default_left = -1.0, default_right = 1.0;
const GLfloat default_bottom = -1.0, default_top = 1.0;
const GLfloat default_zNear = 1.0, default_zFar = 5.0;

const GLfloat  dr = 5.0;

// Viewing transformation parameters
GLfloat viewDist = default_viewDist;
GLfloat az = 0.0;  // angle of deviation from the x-axis
GLfloat el = 0.0;  // elevation angle measured from the ground 

GLuint  view;  // view matrix uniform shader variable location

// Projection transformation parameters (clipping volume)
GLfloat left = default_left,  right = default_right;
GLfloat bottom = default_bottom,  top = default_top;
GLfloat zNear = default_zNear,  zFar = default_zFar;

GLuint  projection; // projection matrix uniform shader variable location

//----------------------------------------------------------------------------

// quad generates two triangles for each face and assigns colors
//    to the vertices

int Index = 0;

void quad( int a, int b, int c, int d )
{
    colors[Index] = vertex_colors[a]; points[Index] = vertices[a]; Index++;
    colors[Index] = vertex_colors[b]; points[Index] = vertices[b]; Index++;
    colors[Index] = vertex_colors[c]; points[Index] = vertices[c]; Index++;
    colors[Index] = vertex_colors[a]; points[Index] = vertices[a]; Index++;
    colors[Index] = vertex_colors[c]; points[Index] = vertices[c]; Index++;
    colors[Index] = vertex_colors[d]; points[Index] = vertices[d]; Index++;
}

//----------------------------------------------------------------------------

// generate 12 triangles: 36 vertices and 36 colors
void colorcube()
{
    quad( 1, 0, 3, 2 );
    quad( 2, 3, 7, 6 );
    quad( 3, 0, 4, 7 );
    quad( 6, 5, 1, 2 );
    quad( 4, 5, 6, 7 );
    quad( 5, 4, 0, 1 );
}

//----------------------------------------------------------------------------

// OpenGL initialization
void init()
{
    colorcube();

    // Create a vertex array object
    GLuint vao;
    glGenVertexArrays( 1, &vao );
    glBindVertexArray( vao );

    // Create and initialize a buffer object
    GLuint buffer;
    glGenBuffers( 1, &buffer );
    glBindBuffer( GL_ARRAY_BUFFER, buffer );
    glBufferData( GL_ARRAY_BUFFER, sizeof(points) + sizeof(colors),
                  NULL, GL_STATIC_DRAW );
    glBufferSubData( GL_ARRAY_BUFFER, 0, sizeof(points), points );
    glBufferSubData( GL_ARRAY_BUFFER, sizeof(points), sizeof(colors), colors );

    // Load shaders and use the resulting shader program
    GLuint program = InitShader( "vshader.glsl", "fshader.glsl" );
    glUseProgram( program );

    // set up vertex arrays
    GLuint vPosition = glGetAttribLocation( program, "vPosition" );
    glEnableVertexAttribArray( vPosition );
    glVertexAttribPointer( vPosition, 4, GL_FLOAT, GL_FALSE, 0,
                           BUFFER_OFFSET(0) );

    GLuint vColor = glGetAttribLocation( program, "vColor" ); 
    glEnableVertexAttribArray( vColor );
    glVertexAttribPointer( vColor, 4, GL_FLOAT, GL_FALSE, 0,
                           BUFFER_OFFSET(sizeof(points)) );

    view = glGetUniformLocation( program, "view" );
    projection = glGetUniformLocation( program, "projection" );
    
    glEnable( GL_DEPTH_TEST );
    glClearColor( 1.0, 1.0, 1.0, 1.0 ); 
}

//----------------------------------------------------------------------------

// for debugging
void printMat(mat4 M)
{
    for (int i=0; i < 4; i++) {
        for (int j=0; j < 4; j++)
            fprintf(stderr, "%10.5f ", M[i][j]);
        fprintf(stderr, "\n");
    }
}

//----------------------------------------------------------------------------

// for debugging
/* We can inspect the 4D clip coordinates and the 3D normalized device coordinates
 * of the 8 vertices of the cube here. The clip coordinates are what we assign
 * to the gl_Position variable in the vertex shader. When any vertex whose clip
 * coordinates (x,y,z,w) do not satisfy all the conditions below:
 *   -w <= x <= w
 *   -w <= y <= w
 *   -w <= z <= w
 * would be clipped away.
 */
void printPoints(mat4 M)
{
    fprintf(stderr, "4D Clip coordinates:\n");
    for (int i=0; i < 8; i++) {
        vec4 p = M*vertices[i];
        fprintf(stderr, "vertex %d: [%10.5f, %10.5f, %10.5f, %10.5f]\n",
            i, p[0], p[1], p[2], p[3]);
    }
    fprintf(stderr, "3D Normalized device coordinates:\n");
    for (int i=0; i < 8; i++) {
        vec4 p = M*vertices[i];
        p = p / p[3];
        fprintf(stderr, "vertex %d: [%10.5f, %10.5f, %10.5f]\n", i, p[0], p[1], p[2]);
    }
    fprintf(stderr, "\n");
}

//----------------------------------------------------------------------------

void display( void )
{
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // Compute the Cartesian coordinates of the camera using the viewDist and the
    // two angles (elevation and azimuth).
    //
    // Note that our x-axis points to the right, y-axis points up, and z-axis points 
    // toward us (this is different from coordinate systems used in many maths books
    // where the z-axis points upward). Also, the elevation angle (el) that we measure
    // here is from the ground (horizon) rather than from the North pole as in some
    // textbooks.
    // DegreesToRadians is a constant defined in Angel.h
    point4  eye( viewDist*cos(az*DegreesToRadians)*cos(el*DegreesToRadians),
                 viewDist*sin(el*DegreesToRadians),
                 viewDist*sin(az*DegreesToRadians)*cos(el*DegreesToRadians),
                 1.0 );
    point4  at( 0.0, 0.0, 0.0, 1.0 ); // the camera fixates at the origin
    vec4    up( 0.0, 1.0, 0.0, 0.0 );

    // view matrix
    mat4  viewMat = LookAt( eye, at, up );
    // fprintf(stderr, "LookAt returns\n");
    // printMat(viewMat);
    // fprintf(stderr, "\n");
    glUniformMatrix4fv( view, 1, GL_TRUE, viewMat );

    mat4  projectionMat = Ortho( left, right, bottom, top, zNear, zFar );
    // fprintf(stderr, "Ortho returns\n");
    // printMat(projectionMat);
    // fprintf(stderr, "\n");
    glUniformMatrix4fv( projection, 1, GL_TRUE, projectionMat );

    // fprintf(stderr, "projection * view = \n");
    // printMat(projectionMat * viewMat);
    // fprintf(stderr, "\n");

    // printPoints(projectionMat * viewMat);

    glDrawArrays( GL_TRIANGLES, 0, NumVertices );

    char title[256];
    sprintf(title, "(l,r,b,t,n,f)=(%1.2f,%1.2f,%1.2f,%1.2f,%1.2f,%1.2f), (%1.3f,%1.3f,%1.3f) ",left,right,bottom,top,zNear,zFar,viewDist,az,el);
    glutSetWindowTitle(title);

    glutSwapBuffers();
}

//----------------------------------------------------------------------------

void keyboard( unsigned char key, int x, int y )
{
    switch( key ) {
    case 033: // Escape Key
    case 'q': case 'Q':
        exit( EXIT_SUCCESS );
        break;

    case 'x': left += 0.1; right -= 0.1; break;
    case 'X': left -= 0.1; right += 0.1; break;
    case 'y': bottom += 0.1; top += 0.1; break;
    case 'Y': bottom -= 0.1; top += 0.1; break;
    case 'z': zNear  += 0.1; zFar -= 0.1; break;
    case 'Z': zNear -= 0.1; zFar += 0.1; break;
    case 'r': viewDist -= 0.1; break;
    case 'R': viewDist += 0.1; break;
    case 'a': az -= dr; if (az < -180) az = -180; break;
    case 'A': az += dr; if (az > 180) az = 180; break;
    case 'e': el -= dr; if (el < -90) el = -90; break;
    case 'E': el += dr; if (el > 90) el = 90; break;

    case ' ':  // reset values to their defaults
        left = default_left;
        right = default_right;
        bottom = default_bottom;
        top = default_top;
        zNear = default_zNear;
        zFar = default_zFar;

        viewDist = default_viewDist;
        az  = 0.0;
        el  = 0.0;
        break;
    }
    glutPostRedisplay();
}

//----------------------------------------------------------------------------

void reshape( int width, int height )
{
    glutReshapeWindow( WIN_WIDTH, WIN_HEIGHT );
}

//----------------------------------------------------------------------------

int main( int argc, char **argv )
{
    glutInit( &argc, argv );
    glutInitDisplayMode( GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH );
    glutInitWindowSize( WIN_WIDTH, WIN_HEIGHT );
    glutInitContextVersion( 3, 2 );
    glutInitContextProfile( GLUT_CORE_PROFILE );
    glutCreateWindow( "Color Cube" );

    glewInit();

    init();

    glutDisplayFunc( display );
    glutKeyboardFunc( keyboard );
    glutReshapeFunc( reshape );

    fprintf(stderr, "Default clipping planes are:\n");
    fprintf(stderr, "(left,right,bottom,top,near,far) = (%1.1f,%1.1f,%1.1f,%1.1f,%1.1f,%1.1f)\n",
        default_left, default_right, default_bottom, default_top,
        default_zNear, default_zFar);
    fprintf(stderr, "Use the following letter to alterate the clipping planes:\n");
    fprintf(stderr, " x  --  left += 0.1; right -= 0.1; (clipping volume narrower)\n");
    fprintf(stderr, " X  --  left -= 0.1; right += 0.1; (clipping volume wider)\n");
    fprintf(stderr, " y  --  bottom += 1.1; top -= 1.1; (clipping volume shorter)\n");
    fprintf(stderr, " Y  --  bottom -= 0.1; top += 0.1; (clipping volume taller)\n");
    fprintf(stderr, " z  --  zNear  += 0.1; zFar -= 0.1; (clipping volume smaller in depth)\n");
    fprintf(stderr, " Z  --  zNear -= 0.1; zFar += 0.1; (clipping volume larger in depth)\n");

    fprintf(stderr, "\n");
    fprintf(stderr, "Use the following letters to change the camera's view point:\n");
    fprintf(stderr, " r  --  viewDist -= 0.1;\n");
    fprintf(stderr, " R  --  viewDist += 0.1;\n");
    fprintf(stderr, " a  --  decrease the azimuth angle by %1.1f deg\n", dr);
    fprintf(stderr, " A  --  increase the azimuth angle by %1.1f deg\n", dr);
    fprintf(stderr, " e  --  decrease the elevation angle by %1.1f deg\n", dr);
    fprintf(stderr, " E  --  increase the elevation angle by %1.1f deg\n", dr);
    fprintf(stderr, "Notes:\n");
    fprintf(stderr, "-180 deg <= azimuth <= 180 deg\n");
    fprintf(stderr, "-90 deg <= elevation <= 90 deg\n");
    
    glutMainLoop();
    return 0;
}
